/* $Id: getdata.c,v 1.137 1999/03/25 03:48:08 ericjb Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

/* This file groups all the functions associated with the getdata
   subsystem.  The vxi registers accessed by the functions are Send
   DATa 1, STATus. */

#include "sema.h"

/*#define	FIFO_COUNT_TEST*/
/*#define	DATA_TEST*/
#define		CVT_TIMEOUT	1.0

#ifdef	E1485_SOURCE
#undef	FIFO_COUNT_TEST
#endif

/* Normally, ilpopfifo handles swapping correctly when passed swap=1,
   which tells it to swap if necessary.  However, byte swapping gets
   complicated when reading 16-bit data as 32-bit words.  Now swap=1
   gets it wrong, so we'd like to set swap=0 and then do the word swap
   ourselves if necessary.  However, the Radisys implementation of
   ilblockcopy ignores the swap flag and always swaps, so that fails.
   Instead, we pass 1 in this case, and then undo the longword swap
   and do our own word swap. */

/* Forward reference of static functions */
static SHORTSIZ16 read_raw_group(E1432ID, SHORTSIZ16, SHORTSIZ16, void *,
				 LONGSIZ32, struct e1432_trailer *,
				 LONGSIZ32 *);
static SHORTSIZ16 read_float32_group(E1432ID, SHORTSIZ16, SHORTSIZ16,
				     FLOATSIZ32 *, LONGSIZ32,
				     struct e1432_trailer *,
				     LONGSIZ32 *);
static SHORTSIZ16 read_float64_group(E1432ID, SHORTSIZ16, SHORTSIZ16,
				     FLOATSIZ64 *, LONGSIZ32,
				     struct e1432_trailer *,
				     LONGSIZ32 *);

static void
compute_blocksize(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 which,
		  LONGSIZ32 *bs, SHORTSIZ16 *data_size)
{
    if (which == E1432_REQUEST_DATA)
    {
        if (mn->rs_magsq)
           *bs = mn->rs_bsize / 2;
        else
           *bs = mn->rs_bsize;
        if (which == E1432_TIME_DATA)
	{
	    *data_size = mn->data_size;
	}
        else
	{
	    *data_size = E1432_DATA_SIZE_FLOAT32;
	}
    }
    else if ( which == E1432_USER1_DATA )
    {
	LONGSIZ32 meas_type;

	(void) i1432_direct_read32_register(mn, E1432_USER1_BLOCKSIZE_REG, bs);
	(void) i1432_direct_read32_register(mn, E1432_MEAS_TYPE_REG,
	  &meas_type);
	if ( meas_type == E1432_MEAS_TYPE_OCTAVE )
	{
	    /* OCTAVE data - 32 bit float */
            *data_size = E1432_DATA_SIZE_FLOAT32;
	}
	else
	{
	    /* USER1 data - 32 bit fixed */
	    *data_size = E1432_DATA_SIZE_32;
	}
    }
    else
    {
        if ( mn->freq_mag_squared
	  && (which == E1432_FREQ_DATA || which == E1432_ORDER_DATA) )
        {
    	    *bs = mn->blocksize / 2;
	}
	else
	{
            *bs = mn->blocksize;
	}
        if( mn->zoom )
	{
	    *bs *= 2;
	}
        if (which == E1432_TIME_DATA)
	{
	    *data_size = mn->data_size;
	}
        else
	{
	    *data_size = E1432_DATA_SIZE_FLOAT32;
	}
    }
}

/*
 *********************************************************************
 Check for lost points, and flush data from all modules in group if
 any have lost points.  Returns 0 if lost points were found, 1 if not,
 negative for error.
 *********************************************************************
 */
static SHORTSIZ16
check_lost_points(E1432ID hw, E1432_GROUP_LIST_NODE *gn)
{
    SHORTSIZ16 error, lost;
    LONGSIZ32 flush;
    int     mod;

    /* check all modules in the group for one with lost points */
    flush = 0;
    for (mod = 0; mod < gn->modcount; mod++)
    {
	/* Read only the top 16 bits of this register, the firmware
	   ensures that it is non-zero if points were lost.  This is
	   faster and less error-prone when using local bus than using
	   all 32 bits. */
	error = i1432_direct_read_register(gn->modlist[mod],
					   E1432_LOST_POINT_REG,
					   &lost);
	if (error)
	    return error;

	if (lost > flush)
	    flush = lost;
    }

    /* if lost points in one module, send command to flush data in all */
    if (flush)
    {
	/* flush master first */
        for (mod = 0; mod < gn->modcount; mod++)
	    if(gn->modlist[mod] == gn->trigmn)
	    {
	        error = i1432_write_cmd1(hw, gn->modlist[mod]->chan_id, 
				     E1432_CMD_FLUSH_SCAN_DATA, flush);
		if (error)
		    return error;
	    }

	/* flush all slaves */
        for (mod = 0; mod < gn->modcount; mod++)
	    if(gn->modlist[mod] != gn->trigmn)
	    {
	        error = i1432_write_cmd1(hw, gn->modlist[mod]->chan_id, 
				     E1432_CMD_FLUSH_SCAN_DATA, flush);
		if (error)
		    return error;
	    }

	i1432_kludge_data_rpm_flag = 0; /* Saved data RPM no longer valid */
	i1432_kludge_enable_flag = 0; /* Ready for next scan */
	i1432_kludge_tach_flag = 1; /* Slave tach used, need new trigger */

	return 0;	/* Lost points found */
    }

    return 1;		/* No lost points found */
}

/*
 *********************************************************************
 Check for modules with enable_none set, and flush data from those
 modules.  Modules set enable_none if they have no data enabled, so
 that the host knows to flush them.  This keeps the modules with no
 data enabled synchronized with other modules that do have data
 enabled.  This function should be called only when the module already
 has said that it has data available.

 Because this function looks at the ENABLE_NONE register anyway, this
 function returns a status value that indicates whether any module in
 the group has data enabled.  The caller (the block_available_group
 function) uses this information.

 Returns 0 if nothing enabled on any module, 1 if something is
 enabled, negative for error.
 ********************************************************************* */
static SHORTSIZ16
check_enable_none(E1432ID hw, E1432_GROUP_LIST_NODE *gn)
{
    SHORTSIZ16 error, enable_none;
    int     mod, all;

    /* Check all modules in the group for enable_none register set */
    all = 1;
    for (mod = 0; mod < gn->modcount; mod++)
    {
	/* Read only the top 16 bits of this register, the firmware
	   ensures that it is non-zero if enable_none is set.  This is
	   faster and less error-prone when using local bus than using
	   all 32 bits. */
	error = i1432_direct_read_register(gn->modlist[mod],
					   E1432_ENABLE_NONE_FLUSH_REG,
					   &enable_none);
	if (error)
	    return error;

	if (enable_none)
	{
	    error = i1432_write_cmd1(hw, gn->modlist[mod]->chan_id,
				     E1432_CMD_FLUSH_SCAN_DATA, 1);
	    if (error)
		return error;
	    /* We only get here if there was a block available and
	       we've told modules with no enabled channels to flush
	       the scan.  At this point, these modules no longer say
	       they have a block available.  Normally, the host will
	       now read data from the other modules and it won't
	       matter.  But if the host doesn't read the data, and
	       instead checks block available again, we don't want
	       these modules to make block_available return false.
	       This flag indicates that we must handle that. */
	    i1432_kludge_enable_flag = 1; /* Already processed enable_none */
	}
	else
	    all = 0;
    }

    if (all)
    {
	i1432_kludge_data_rpm_flag = 0; /* Saved data RPM no longer valid */
	i1432_kludge_enable_flag = 0; /* Ready for next scan */
	i1432_kludge_tach_flag = 1; /* Slave tach used, need new trigger */
	return 0;	/* Nothing enabled */
    }

    return 1;		/* Something enabled */
}

/*
 *********************************************************************
 Return group block status
 *********************************************************************
 */
static SHORTSIZ16
block_available_group(E1432ID hw, SHORTSIZ16 groupID)
{
    SHORTSIZ16 error, rv;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    int     type, count;

    TRACE_PRINTF(2, ("    block_available_group(0x%p, %d)\n",
		     hw, groupID));

    /* get e1432_group_list_node */
    gn = i1432_get_group_node(hw, groupID);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);

    /* Start by assuming data is available */
    rv = 1;

    cn = gn->chanlist;
    /* check each channel */
    if (!i1432_kludge_enable_flag)
	while (cn)
	{
	    /* Get the channel type and count */
	    type = (cn->chanID & E1432_CHAN_TYPE_MASK) >>
		E1432_CHAN_TYPE_SHIFT;
	    count = (cn->chanID & E1432_CHAN_MASK) - 1;

	    /* Check active input channels */
	    if (type == E1432_CHAN_TYPE_INPUT &&
		i1432_chan_list[type][count].active == E1432_CHANNEL_ON)
	    {
		/* Formerly we would check only enabled channels.
		   Instead, now we check all channels, and later check
		   the ENABLE_NONE register to see if a module
		   actually has any data enabled.  This should be
		   faster, because we avoid all the logic to check
		   enable flags.

		   Because we now do e1432_block_available even if no
		   data is enabled, we will notice FIFO overflow or
		   other module errors, even if no data is enabled, so
		   there is no need for a separate loop below in the
		   case of no data enabled. */

		/* Check this channel for data */
		error = e1432_block_available(hw, cn->chanID);
		if (error < 0)
		    return error;
		if (error == 0)
		{
		    /* Data is not available.  But don't just return
		       zero here, because we need to check for lost
		       points below.  Not sure why, but Doug had some
		       problem with averaging and order tracking that
		       this might have fixed. */
		    rv = 0;
		    break;
		}
	    }

	    /* Check active tach channels */
	    if (type == E1432_CHAN_TYPE_TACH &&
		i1432_chan_list[type][count].active == E1432_CHANNEL_ON)
	    {
		/* When doing rpm arming or order tracking with
		   multiple modules, we send tach and trigger times
		   from the trigger master to the slaves.  The master
		   makes the tach and trigger times available once it
		   triggers, so the master almost immediately asserts
		   block available.  But the slaves will not be ready
		   for a short while.  As long as the slaves have
		   input channels, that's OK because the slave won't
		   assert block available until it is really ready.

		   But if the slave has no active input channels (just
		   tach channels), then we aren't going to be checking
		   block available in the slaves.  Nevertheless, we
		   don't want to say that a block is available until
		   the slaves have the correct RPM values.  So we use
		   a flag (set by e1432_send_trigger) to tell if the
		   slave tach channels have the correct RPM values
		   ready.

		   This flag is a kludge.  At the very least, it
		   should have been a field in the group list node.
		   Better would have been to have a real register on
		   the slave modules that e1432_block_available could
		   check.  But using a register in the module is hard
		   because you have to unwind all of the rpm arming
		   and order tracking firmware to figure out where to
		   set and clear the register.  So instead we keep a
		   flag in the host code.

		   A flag value zero means that the slave tach channel
		   is ready to go.  The flag is set to one when the
		   host reads data from any module, indicating that
		   the slave tach channel now holds "old" RPM values
		   and we need a new trigger before we consider the
		   slave module to be ready again.  The flag is set to
		   two when the host sees a new trigger in the master
		   and sends it to the slave.  The flag is set back to
		   zero when its value is two and the slaves all say
		   they are done processing the trigger.

		   You can't make the flag a field of the group list
		   node because it must be set from
		   e1432_read_xxx_data, which is often only given a
		   channel ID. */
		if (gn->modcount > 1 && gn->otra)
		    if (i1432_kludge_tach_flag != 0)
		    {
			/* Originally we just did a return 0 here, but
			   it seems like we need to copy the input
			   channel code above and allow for checking
			   for lost points below. */
			rv = 0;
			break;
		    }
	    }

	    /* next channel */
	    cn = cn->next;
	}

    /* This checks for lost points.  If one is found, then we flush
       that scan.  In this case, we return zero because data is not
       available anymore, even if the rv value from above is one.  The
       check for lost points should come after checking for data
       available, otherwise there's a window between them where the
       module might set data avaiable and lost point, and we'd see
       only the data available and things would not work correctly. */
    error = check_lost_points(hw, gn);
    if (error <= 0)
	return error;	/* Error, or lost points found */

    if (rv)
    {
	/* This checks for modules that have no data enabled.  They
	   will not do FIFO_cleanup_out themselves, because no data
	   will get read from them.  So we instead tell them to flush
	   the scan.  We only do this when we get block available from
	   all modules, so we keep the modules in sync.

	   Do this after checking for lost points, to avoid problems
	   if both lost points and enable none are set. */
	error = check_enable_none(hw, gn);
	if (error <= 0)
	    return error;	/* Error, or nothing enabled */
    }

    /* Return rv indicating whether data is available */
    return rv;
}

/*
 *********************************************************************
 Return positive number greater than 0 if the block available flag on
 the module for this channel is set.  Return 0 if no block available.
 Return negative error if error.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_block_available(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error, val;

    TRACE_PRINTF(0, ("e1432_block_available(0x%p, %d)\n",
		     hw, ID));
    /* Check ID */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;
    if (ID <= 0)
	return block_available_group(hw, ID);

    /* There is minor subtlety here.

       The BLOCK_READY bit is set when a scan gets ready, and stays
       set until a short time after then end of transferring that
       scan.  On a fast host computer, there is always the possibility
       that the bit will still be set after it finishes transferring
       all the data in a scan, so we can't just rely on BLOCK_READY to
       tell us that a block of data is ready.

       We CAN rely on the DATA_AVAIL register, which is set by the
       substrate after we enable DMA to the host, and is cleared by
       the host before we actually transfer the data.  So, that's what
       we use to determine if there is really a block of data ready.

       For detecting FIFO overflow, we can't just use the MEAS_ERROR
       bit, because we don't want to flag an overflow until after the
       FIFO gets emptied.  We can't just qualify this with the
       DATA_AVAIL register we use above, since DATA_AVAIL is briefly
       zero between blocks in a scan, and we don't want to flag an
       overflow at that time.  We CAN qualify it with the BLOCK_READY
       bit, since BLOCK_READY will stay set until the FIFO empties.

       But wait, there's more.  If we're doing FFTs or averaging, the
       BLOCK_READY bit really can get cleared for brief times after a
       FIFO overflow but before the FIFO empties.  So BLOCK_READY
       won't work for FIFO overflow detection either.  Instead, I have
       created a new FIFO_EMPTIED bit in IRQ_STATUS2 that indicates
       both FIFO overflow and FIFO emptied.  All eight normal bits for
       IRQs are already taken, so there is no way to interrupt on
       FIFO_EMPTIED. */

    error = e1432_read_register(hw, ID, E1432_DATA_AVAIL_REG, &val);
    if (error)
	return error;

    /* If no data available, check for errors */
    if (val == 0)
    {
        error = e1432_read_register(hw, ID, E1432_IRQ_STATUS2_REG, &val);
        if (error)
	    return error;

        /* If tach overflow and no more data, return TACH_OVERFLOW error */
        if ((val & E1432_IRQ_MEAS_ERROR) && (val & E1432_STATUS2_TACH_OVERFLOW))
	    return i1432_id_print_error(hw, ID, ERR1432_TACH_BUFFER_OVERFLOW);

	/* If FIFO_EMPTIED bit is set, return FIFO_OVERRUN error */
	if ((val & E1432_STATUS2_FIFO_EMPTIED) != 0)
	    return i1432_id_print_error(hw, ID, ERR1432_FIFO_OVERRUN);

        /* Check for module errors */
        return i1432_error_check(hw, ID);
    }

    return 1;
}

/*ARGSUSED*/
static SHORTSIZ16
xfer(E1432ID hw, SHORTSIZ16 ID, unsigned long *lbuf,
     unsigned long size, int data_is_32)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    unsigned long dma_length, length;
#ifdef	FIFO_COUNT_TEST
    SHORTSIZ16 fifo_count1, fifo_count2;
#endif

    /* get module for this channel */
    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;

    dma_length = size;

    while (dma_length > 0)
    {
	length = dma_length;
	if (length > E1432_HW_FIFO_SIZE / 4)
	    length = E1432_HW_FIFO_SIZE / 4;

#ifdef	FIFO_COUNT_TEST
	error = i1432_direct_read_register(mn,
					   E1432_FIFO_COUNT_REG,
					   &fifo_count1);
	if (error)
	    return error;
#endif

	error = i1432_xfer_block(mn, (unsigned long *) 0,
				 lbuf, length, 1, 1);
	if (error)
	    return error;

#ifdef	FIFO_COUNT_TEST
	error = i1432_direct_read_register(mn,
					   E1432_FIFO_COUNT_REG,
					   &fifo_count2);
	if (error)
	    return error;
#ifndef	DATA_TEST
	if (((fifo_count1 - fifo_count2) & 0xffff) != length)
#endif
	{
	    int     i, error_count, print_anyway, match_count;
	    LONGSIZ32 dma_start, value, value2;

	    if (((fifo_count1 - fifo_count2) & 0xffff) != length)
	    {
		(void) fprintf(stderr,
			       "fifo_count1, fifo_count2 = "
			       "0x%4.4x, 0x%4.4x\n",
			       (unsigned short) fifo_count1,
			       (unsigned short) fifo_count2);
		(void) fprintf(stderr,
			       "length, fifo_count1 - fifo_count2 = "
			       "0x%4.4x, 0x%4.4x\n",
			       length, fifo_count1 - fifo_count2);
	    }

	    error = i1432_direct_read32_register(mn,
						 E1432_DMA_START_REG,
						 &dma_start);
	    if (error)
		return error;

	    error_count = 0;
	    match_count = 0;
	    print_anyway = 0;
	    for (i = 0; i < length; i++)
	    {
		error = e1432_read_dsp_mem(hw, ID, 0,
					   dma_start + i, &value);
		if (error)
		    return error;
		if (value != lbuf[i])
		{
		    error_count++;
		    if (!print_anyway && error_count > 10 &&
			i < length - 10)
			continue;
		    if (print_anyway)
			print_anyway = 0;
		    (void) fprintf(stderr,
				   "data mismatch at %d, got "
				   "0x%8.8x, should be 0x%8.8x\n",
				   i, lbuf[i], value);
		}
	    }
	    if (error_count > 0)
	    {
		(void) fprintf(stderr,
			       "total errors 0x%x\n",
			       error_count);
		exit(1);
	    }
	}
#endif	/* FIFO_COUNT_TEST */

#if SWAP
	if (data_is_32 != mn->d32)
	{
	    error = i1432_two_four_swap((char *) lbuf, length * 4);
	    if (error)
		return error;
	}
#endif

	lbuf += length;
	dma_length -= length;
    }

    return 0;
}


/*
 *********************************************************************
 Read a block of raw data from channel ChanID.

 If append status is on, calling this function should be followed by
 a call to e1432_read_status_data in order to "clean" the fifo,
 even if this status data is not needed.

 Return 0 if OK, else return negative error number.

 This function should be called as many times (in sequence) as the
 number of active channels on a module. This is due to the fact that
 the module outputs data through a FIFO, one channel block at a time.
 Each time the 'block ready' flag is set, it means that at least a block
 of data is available on each of the active channels.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_read_raw_data(E1432ID hw, SHORTSIZ16 chanID, SHORTSIZ16 which,
		    void *voidbuf, LONGSIZ32 size,
		    struct e1432_trailer *trailer,
		    LONGSIZ32 * actualCnt)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, data_size;
    LONGSIZ32 maxsize, xfer_data_pts, bs;
    unsigned short *buf;
    unsigned long *lbuf;
    int     separate_trailer, type, count;
#if SWAP
    LONGSIZ32 data_pts;
#endif

    TRACE_PRINTF(0, ("e1432_read_raw_data"
		     "(0x%p, %d, %d, 0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, chanID, which, voidbuf, size, trailer,
		     actualCnt));

    buf = (unsigned short *) voidbuf;
    lbuf = (unsigned long *) voidbuf;

    /* check if group read */
    if (chanID < 0)
	return read_raw_group(hw, chanID, which, buf, size,
			      trailer, actualCnt);
    /* check if real channel */
    error = i1432_checkID(hw, chanID);
    if (error)
	return error;
    /* must be a channel that has a module allocated */
    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;

    type = (chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
    count = (chanID & E1432_CHAN_MASK) - 1;
    if (i1432_chan_list[type][count].active != E1432_CHANNEL_ON)
	return i1432_print_error(ERR1432_NO_CHANNEL);

    if (mn->data_size == E1432_DATA_SIZE_16 &&
	((size & 0x1) != 0 ||	/* Odd number of samples */
	 (mn->d32 && ((long) lbuf & 0x3) != 0))) /* Misaligned buffer */
	return i1432_print_error(ERR1432_DATA_MISALIGNED);

    compute_blocksize(mn, which, &bs, &data_size);
    if (data_size == E1432_DATA_SIZE_FLOAT64) bs *= 2;  /* doubles */

    /* Set read size to lessor of the blocksize and buffer size (so
       that no more than one block is read and buffer is not overrun).
       If append_status is on, allow room for the trailer data as
       well. */
    maxsize = bs;
    if (mn->append_status == E1432_APPEND_STATUS_ON)
    {
	if (data_size == E1432_DATA_SIZE_16)
	    maxsize += 16;
	else
	    maxsize += 8;
    }
    if (size > maxsize)
	size = maxsize;

    xfer_data_pts = size;
    if (data_size == E1432_DATA_SIZE_16)
    {
	xfer_data_pts /= 2;
    }

    separate_trailer = (trailer != NULL && 
			mn->append_status == E1432_APPEND_STATUS_ON);

    i1432_kludge_data_rpm_flag = 0; /* Saved data RPM no longer valid */
    i1432_kludge_enable_flag = 0; /* Ready for next scan */
    i1432_kludge_tach_flag = 1;	/* Slave tach used, need new trigger */

    error = i1432_data_handshake(mn);
    if (error)
	return error;

    /* Transfer data */
    error = xfer(hw, chanID, lbuf, xfer_data_pts,
		 data_size != E1432_DATA_SIZE_16);
    if (error)
	return error;

    if (separate_trailer)
    {
	error = xfer(hw, chanID, (unsigned long *) trailer,
		     sizeof *trailer / sizeof (LONGSIZ32), 1);
	if (error)
	    return error;
    }

    /* The minimum amount of data transferred per channel is four
       words, so that the block is bigger than the hardware pipeline
       in the module.  If the block were to fit completely in the
       pipeline, the data handshake would get messed up. */
    if (xfer_data_pts < 4 && !separate_trailer)
    {
	unsigned long dummy[4];
	error = xfer(hw, chanID, dummy, 4 - xfer_data_pts, 1);
	if (error)
	    return error;
    }

    *actualCnt = size;

#if SWAP
    if (data_size == E1432_DATA_SIZE_16)
    {
	data_pts = (bs > size) ? size : bs;

	/* Data OK, trailer wrong */
	error = i1432_two_four_swap((char *) buf + data_pts,
				    (size - data_pts) * 2);
	if (error)
	    return error;
    }
#endif

    return 0;
}

/*
 *********************************************************************
 Read all channels in group and stuff into buffer.  All channels of
 data are assumed to be ready to read.  Data is place in the buffer in
 the order of chanlist from create_channel_group.  Return 0 if OK,
 else return negative error number.
 *********************************************************************
 */
static SHORTSIZ16
read_raw_group(E1432ID hw, SHORTSIZ16 groupID, SHORTSIZ16 which,
	       void *voidbuf, LONGSIZ32 size,
	       struct e1432_trailer *trailer, LONGSIZ32 * actualCnt)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_MODULE_LIST_NODE *mn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 *buf;
    SHORTSIZ16 error, data_size;
    LONGSIZ32 bufferoffset, actcnt, acttotal, bufferleft;
    LONGSIZ32 xfersize, bs;
    int     type, count;

    TRACE_PRINTF(2, ("    read_raw_group"
		     "(0x%p, %d, %d, 0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, groupID, which, voidbuf, size, trailer,
		     actualCnt));

    /* convert given buffer to 16 bit buffer */
    buf = (SHORTSIZ16 *) voidbuf;
    /* check group ID */
    error = i1432_checkID(hw, groupID);
    if (error)
	return error;
    /* get e1432_group_list_node */
    gn = i1432_get_group_node(hw, groupID);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);
    cn = gn->chanlist;

    bufferoffset = 0;
    acttotal = 0;
    bufferleft = size;
    /* read each channel */
    while (cn && (bufferleft > 0))
    {
	/* read data for active channels */
	type = (cn->chanID & E1432_CHAN_TYPE_MASK) >>
	    E1432_CHAN_TYPE_SHIFT;
	count = (cn->chanID & E1432_CHAN_MASK) - 1;
	if (i1432_chan_list[type][count].active == E1432_CHANNEL_ON)
	{
	    /* get module node */
	    error = i1432_get_module_from_chan(hw, cn->chanID, &mn);
	    if (error)
		return error;

            compute_blocksize(mn, which, &bs, &data_size);

	    xfersize = bufferleft;
	    if (trailer != NULL && xfersize > bs)
		xfersize = bs;

	    /* get this channel of data */
	    error = e1432_read_raw_data(hw, cn->chanID, which,
					buf + bufferoffset,
					xfersize, trailer, &actcnt);
	    if (error)
		return error;
	    /* total up bytes transfered */
	    acttotal += actcnt;
	    /* subtract used buffer space */
	    bufferleft -= actcnt;
	    /* If separate trailers, increment to next */
	    if (trailer != NULL)
		trailer++;
	    /* bump buffer offset to next empty space */
	    bufferoffset +=
		actcnt * ((data_size == E1432_DATA_SIZE_16) ? 1 : 2);
	}
	/* next channel */
	cn = cn->next;
    }
    /* report total number of data points read */
    *actualCnt = acttotal;
    return 0;
}

/*
 *********************************************************************
 Copy of read raw data with scaling added.  Return 0 if OK, else
 return negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_read_float32_data(E1432ID hw, SHORTSIZ16 chanID, SHORTSIZ16 which,
			FLOATSIZ32 * buf, LONGSIZ32 size,
			struct e1432_trailer *trailer,
			LONGSIZ32 * actualCnt)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, data_size;
    register FLOATSIZ32 scale;
    register FLOATSIZ32 *fptr;
    LONGSIZ32 maxsize, data_pts, bs;
    LONGSIZ32 xfer_data_pts, xfer_trailer_pts;
    register short  *sptr;
    long    i;
    int     separate_trailer, type, count;

    TRACE_PRINTF(0, ("e1432_read_float32_data"
		     "(0x%p, %d, %d, 0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, chanID, which, buf, size, trailer,
		     actualCnt));

    /* check if group read */
    if (chanID < 0)
	return read_float32_group(hw, chanID, which, buf, size,
				  trailer, actualCnt);
    /* check if real channel */
    error = i1432_checkID(hw, chanID);
    if (error)
	return error;
    /* must be a channel that has a module allocated */
    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;

    type = (chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
    count = (chanID & E1432_CHAN_MASK) - 1;
    if (i1432_chan_list[type][count].active != E1432_CHANNEL_ON)
	return i1432_print_error(ERR1432_NO_CHANNEL);

    compute_blocksize(mn, which, &bs, &data_size);

    if (data_size == E1432_DATA_SIZE_16 && (size & 0x1) != 0)
	return i1432_print_error(ERR1432_IO);

    if (data_size != E1432_DATA_SIZE_FLOAT32)
	/* Get scaling for this channel */
	scale = (FLOATSIZ32) i1432_chan_list[type][count].scale;
    else
	scale = 1.0f;

    /* Set read size to lessor of the blocksize and buffer size (so
       that no more than one block is read and buffer is not overrun).
       If append_status is on, allow room for the trailer data as
       well. */
    maxsize = bs;
    if (mn->append_status == E1432_APPEND_STATUS_ON)
    {
	if (data_size == E1432_DATA_SIZE_16)
	    maxsize += 16;
	else
	    maxsize += 8;
    }
    if (size > maxsize)
	size = maxsize;

    data_pts = (bs > size) ? size : bs;

    xfer_data_pts = data_pts;
    xfer_trailer_pts = size - data_pts;
    if (data_size == E1432_DATA_SIZE_16)
    {
	xfer_data_pts /= 2;
	xfer_trailer_pts /= 2;
    }

    separate_trailer = (trailer != NULL &&
			mn->append_status == E1432_APPEND_STATUS_ON);

    i1432_kludge_data_rpm_flag = 0; /* Saved data RPM no longer valid */
    i1432_kludge_enable_flag = 0; /* Ready for next scan */
    i1432_kludge_tach_flag = 1;	/* Slave tach used, need new trigger */

    error = i1432_data_handshake(mn);
    if (error)
	return error;

    /* Get data, but not trailer */
    error = xfer(hw, chanID, (unsigned long *) buf,
		 xfer_data_pts, data_size != E1432_DATA_SIZE_16);
    if (error)
	return error;

    /* Transfer trailer */
    if (separate_trailer)
	error = xfer(hw, chanID, (unsigned long *) trailer,
		     sizeof *trailer / sizeof (LONGSIZ32), 1);
    else
	error = xfer(hw, chanID, (unsigned long *) (buf + data_pts),
		     xfer_trailer_pts, 1);
    if (error)
	return error;
    *actualCnt = size;

    /* The minimum amount of data transferred per channel is four
       words, so that the block is bigger than the hardware pipeline
       in the module.  If the block were to fit completely in the
       pipeline, the data handshake would get messed up. */
    if (xfer_data_pts + xfer_trailer_pts < 4 && !separate_trailer)
    {
	unsigned long dummy[4];
	error = xfer(hw, chanID, dummy,
		     4 - xfer_data_pts - xfer_trailer_pts, 1);
	if (error)
	    return error;
    }

    /* Convert all data, but not trailer */
    if (data_size == E1432_DATA_SIZE_16)
    {
	fptr = buf + data_pts;
	sptr = ((short *) buf) + data_pts;
	for (i = 0; i < data_pts; i++)
	    *--fptr = (FLOATSIZ32) (*--sptr) * scale;
    }
    else if (data_size == E1432_DATA_SIZE_32)
	for (i = 0; i < data_pts; i++, buf++)
	    *buf = (FLOATSIZ32) (*(long *) buf * scale);
    else if (data_size == E1432_DATA_SIZE_32_SERV)
	for (i = 0; i < data_pts; i++, buf++)
	{
	    *(unsigned long *) buf = *(unsigned long *) buf &
		E1432_DATA_SIZE_32_MASK;
	    *buf = (FLOATSIZ32) (*(long *) buf * scale);
	}
    /* Nothing to do for E1432_DATA_SIZE_FLOAT32 */

    return 0;
}

/*
 *********************************************************************
 Read all channels in group and stuff into buffer.  All channels of
 data are assumed to be ready to read.  Data is place in the buffer in
 the order of chanlist from create_channel_group.  Return 0 if OK,
 else return negative error number.
 *********************************************************************
 */
static SHORTSIZ16
read_float32_group(E1432ID hw, SHORTSIZ16 groupID, SHORTSIZ16 which,
		   FLOATSIZ32 * buf, LONGSIZ32 size,
		   struct e1432_trailer *trailer,
		   LONGSIZ32 * actualCnt)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, data_size;
    LONGSIZ32 bufferoffset, actcnt, acttotal, bufferleft;
    LONGSIZ32 xfersize, bs;
    int     type, count;

    TRACE_PRINTF(2, ("    read_float32_group"
		     "(0x%p, %d, %d, 0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, groupID, which, buf, size, trailer,
		     actualCnt));

    /* check group ID */
    error = i1432_checkID(hw, groupID);
    if (error)
	return error;
    /* get e1432_group_list_node */
    gn = i1432_get_group_node(hw, groupID);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);

    cn = gn->chanlist;
    bufferoffset = 0;
    acttotal = 0;
    bufferleft = size;
    /* read each channel */
    while (cn && bufferleft > 0)
    {
	/* read data for active channels */
	type = (cn->chanID & E1432_CHAN_TYPE_MASK) >>
	    E1432_CHAN_TYPE_SHIFT;
	count = (cn->chanID & E1432_CHAN_MASK) - 1;
	if (i1432_chan_list[type][count].active == E1432_CHANNEL_ON)
	{
	    /* get module node */
	    error = i1432_get_module_from_chan(hw, cn->chanID, &mn);
	    if (error)
		return error;

            compute_blocksize(mn, which, &bs, &data_size);

	    xfersize = bufferleft;
	    if (trailer != NULL && xfersize > bs)
		xfersize = bs;

	    /* get this channel of data */
	    error = e1432_read_float32_data(hw, cn->chanID, which,
					    &(buf[bufferoffset]),
					    xfersize, trailer,
					    &actcnt);
	    if (error)
		return error;
	    /* total up bytes transfered */
	    acttotal += actcnt;
	    /* subtract used buffer space */
	    bufferleft -= actcnt;
	    /* If separate trailers, increment to next */
	    if (trailer != NULL)
		trailer++;
	    /* bump buffer offset to next empty space */
	    bufferoffset += actcnt;
	}
	/* next channel */
	cn = cn->next;
    }
    /* report total number of data points read */
    *actualCnt = acttotal;
    return 0;
}

/*
 *********************************************************************
 Copy of read raw data with scaling added.  Return 0 if OK, else
 return negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_read_float64_data(E1432ID hw, SHORTSIZ16 chanID, SHORTSIZ16 which,
			FLOATSIZ64 * buf, LONGSIZ32 size,
			struct e1432_trailer *trailer,
			LONGSIZ32 * actualCnt)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, data_size;
    FLOATSIZ64 scale;
    FLOATSIZ64 *dptr;
    LONGSIZ32 maxsize, data_pts;
    LONGSIZ32 xfer_data_pts, xfer_trailer_pts, bs;
    short  *sptr;
    float  *fptr;
    long   *lptr;
    long    i;
    int     separate_trailer, type, count;

    TRACE_PRINTF(0, ("e1432_read_float64_data"
		     "(0x%p, %d, %d, 0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, chanID, which, buf, size, trailer,
		     actualCnt));

    /* check if group read */
    if (chanID < 0)
	return read_float64_group(hw, chanID, which, buf, size,
				  trailer, actualCnt);
    /* check if real channel */
    error = i1432_checkID(hw, chanID);
    if (error)
	return error;
    /* must be a channel that has a module allocated */
    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;

    type = (chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
    count = (chanID & E1432_CHAN_MASK) - 1;
    if (i1432_chan_list[type][count].active != E1432_CHANNEL_ON)
	return i1432_print_error(ERR1432_NO_CHANNEL);

    compute_blocksize(mn, which, &bs, &data_size);

    if (data_size == E1432_DATA_SIZE_16 && (size & 0x1) != 0)
	return i1432_print_error(ERR1432_IO);

    if (data_size != E1432_DATA_SIZE_FLOAT32
	&& data_size != E1432_DATA_SIZE_FLOAT64)
	/* Get scaling for this channel */
	scale = i1432_chan_list[type][count].scale;

    /* Set read size to lessor of the blocksize and buffer size (so
       that no more than one block is read and buffer is not overrun).
       If append_status is on, allow room for the trailer data as
       well. */
    maxsize = bs;
    if (mn->append_status == E1432_APPEND_STATUS_ON)
    {
	if (data_size == E1432_DATA_SIZE_16)
	    maxsize += 16;
        else if (data_size == E1432_DATA_SIZE_FLOAT64)
	    maxsize += 4;
	else
	    maxsize += 8;
    }
    if (size > maxsize)
	size = maxsize;

    data_pts = (bs > size) ? size : bs;

    xfer_data_pts = data_pts;
    xfer_trailer_pts = size - data_pts;
    if (data_size == E1432_DATA_SIZE_16)
    {
	xfer_data_pts /= 2;
	xfer_trailer_pts /= 2;
    }
    else if ( data_size == E1432_DATA_SIZE_FLOAT64 )
    {
	xfer_data_pts *= 2;
	xfer_trailer_pts *= 2;  /* not sure about this */
    }

    separate_trailer = (trailer != NULL && 
			mn->append_status == E1432_APPEND_STATUS_ON);

    i1432_kludge_data_rpm_flag = 0; /* Saved data RPM no longer valid */
    i1432_kludge_enable_flag = 0; /* Ready for next scan */
    i1432_kludge_tach_flag = 1;	/* Slave tach used, need new trigger */

    error = i1432_data_handshake(mn);
    if (error)
	return error;

    /* Get data, but not trailer */
    error = xfer(hw, chanID, (unsigned long *) buf,
		 xfer_data_pts,
		 data_size != E1432_DATA_SIZE_16);
    if (error)
	return error;

    if (separate_trailer)
	error = xfer(hw, chanID, (unsigned long *) trailer,
		     sizeof *trailer / sizeof (LONGSIZ32), 1);
    else
	error = xfer(hw, chanID, (unsigned long *) (buf + data_pts),
		     xfer_trailer_pts, 1);
    if (error)
	return error;

    /* The minimum amount of data transferred per channel is four
       words, so that the block is bigger than the hardware pipeline
       in the module.  If the block were to fit completely in the
       pipeline, the data handshake would get messed up. */
    if (xfer_data_pts + xfer_trailer_pts < 4 && !separate_trailer)
    {
	unsigned long dummy[4];
	error = xfer(hw, chanID, dummy,
		     4 - xfer_data_pts - xfer_trailer_pts, 1);
	if (error)
	    return error;
    }

    *actualCnt = size;

    /* Convert all data, but not trailer */
    if (data_size == E1432_DATA_SIZE_16)
    {
	dptr = buf + data_pts;
	sptr = ((short *) buf) + data_pts;
	for (i = 0; i < data_pts; i++)
	    *--dptr = *--sptr * scale;
    }
    else if (data_size == E1432_DATA_SIZE_32)
    {
	dptr = buf + data_pts;
	lptr = ((long *) buf) + data_pts;
	for (i = 0; i < data_pts; i++)
	    *--dptr = *--lptr * scale;
    }
    else if (data_size == E1432_DATA_SIZE_32_SERV)
    {
	dptr = buf + data_pts;
	lptr = ((long *) buf) + data_pts;
	for (i = 0; i < data_pts; i++)
	{
	    --lptr;
	    *lptr = *lptr & E1432_DATA_SIZE_32_MASK;
	    *--dptr = *lptr * scale;
	}
    }
    else if (data_size == E1432_DATA_SIZE_FLOAT32)
    {
	dptr = buf + data_pts;
	fptr = ((float *) buf) + data_pts;
	for (i = 0; i < data_pts; i++)
	    *--dptr = *--fptr;
    }
#if SWAP
    else if (data_size == E1432_DATA_SIZE_FLOAT64)
    {
        /* The 64-bit doubles were transferred in 32-bit chunks, so
	   need 32-bit swap. */
	long tmp, *tptr;
	lptr = (long *) buf;
	for (i = 0; i < data_pts; i++)
	{
	    tptr = lptr;
	    tmp = *lptr++;
	    *tptr = *lptr;
	    *lptr++ = tmp;
	}
    }
#endif
    /* no processing of E1432_DATA_SIZE_FLOAT64 otherwise */

    return 0;
}

/*
 *********************************************************************
 Read all channels in group and stuff into buffer.  All channels of
 data are assumed to be ready to read.  Data is place in the buffer in
 the order of chanlist from create_channel_group.  Return 0 if OK,
 else return negative error number.
 *********************************************************************
 */
static SHORTSIZ16
read_float64_group(E1432ID hw, SHORTSIZ16 groupID, SHORTSIZ16 which,
		   FLOATSIZ64 * buf, LONGSIZ32 size,
		   struct e1432_trailer *trailer,
		   LONGSIZ32 * actualCnt)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, data_size;
    LONGSIZ32 bufferoffset, actcnt, acttotal, bufferleft;
    LONGSIZ32 xfersize, bs;
    int     type, count;

    TRACE_PRINTF(2, ("    read_float64_group"
		     "(0x%p, %d, %d, 0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, groupID, which, buf, size, trailer,
		     actualCnt));

    /* check group ID */
    error = i1432_checkID(hw, groupID);
    if (error)
	return error;
    /* get e1432_group_list_node */
    gn = i1432_get_group_node(hw, groupID);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);

    cn = gn->chanlist;
    bufferoffset = 0;
    acttotal = 0;
    bufferleft = size;
    /* read each channel */
    while (cn && bufferleft > 0)
    {
	/* read data for active channels */
	type = (cn->chanID & E1432_CHAN_TYPE_MASK) >>
	    E1432_CHAN_TYPE_SHIFT;
	count = (cn->chanID & E1432_CHAN_MASK) - 1;
	if (i1432_chan_list[type][count].active == E1432_CHANNEL_ON)
	{
	    /* get module node */
	    error = i1432_get_module_from_chan(hw, cn->chanID, &mn);
	    if (error)
		return error;

            compute_blocksize(mn, which, &bs, &data_size);

	    xfersize = bufferleft;
	    if (trailer != NULL && xfersize > bs)
		xfersize = bs;

	    /* get this channel of data */
	    error = e1432_read_float64_data(hw, cn->chanID, which,
					    &(buf[bufferoffset]),
					    xfersize, trailer,
					    &actcnt);
	    if (error)
		return error;
	    /* total up bytes transfered */
	    acttotal += actcnt;
	    /* subtract used buffer space */
	    bufferleft -= actcnt;
	    /* If separate trailers, increment to next */
	    if (trailer != NULL)
		trailer++;
	    /* bump buffer offset to next empty space */
	    bufferoffset += actcnt;
	}
	/* next channel */
	cn = cn->next;
    }
    /* report total number of data points read */
    *actualCnt = acttotal;
    return 0;
}

/*
 *********************************************************************
 Read overflow registers for a group or channel.  Returns 0 if OK,
 otherwise returns negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_check_overloads(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 *any,
		      SHORTSIZ16 *comm, SHORTSIZ16 *diff,
		      SHORTSIZ16 *half)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    LONGSIZ32 bits;
    SHORTSIZ16 error, ovld;
    int     cindex, type, count;

    TRACE_PRINTF(0, ("e1432_check_overloads(0x%p, %d, 0x%p, "
		     "0x%p, 0x%p, 0x%p)\n",
		     hw, ID, any, comm, diff, half));

    /* check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* check if channel or group */
    if (ID < 0)
    {
	/* iterate thru group clear group any overload flag */
	if (any)
	    *any = 0;
	/* iterate thru group */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	while (cn)
	{
	    type = (cn->chanID & E1432_CHAN_TYPE_MASK) >>
		E1432_CHAN_TYPE_SHIFT;
	    count = (cn->chanID & E1432_CHAN_MASK) - 1;
	    if (i1432_chan_list[type][count].active ==
		E1432_CHANNEL_ON)
	    {
		error = e1432_check_overloads(hw, cn->chanID, any,
					      comm, diff, half);
		if (error)
		    return error;
		/* advance ovld flags to next location */
		if (comm)
		    comm++;
		if (diff)
		    diff++;
		if (half)
		    half++;
	    }
	    /* next channel */
	    cn = cn->next;
	}
    }
    else
    {
	/* check one channel overloads */
	cindex = i1432_get_chan_index(hw, ID);
	error = e1432_read32_register(hw, ID, E1432_OVLD_INFO_CREG +
				      cindex * E1432_CREG_OFFSET,
				      &bits);
	if (error)
	    return error;

	/* check common mode */
	ovld = (SHORTSIZ16) ((bits & E1432_FLAG_OVERLOAD_COMM) ? 1 : 0);
	if (comm)
	    *comm = ovld;
	if (any)
	    *any |= ovld;	/* update anyflag */
	/* check differential */
	ovld = (SHORTSIZ16) ((bits & E1432_FLAG_OVERLOAD_DIFF) ? 1 : 0);
	if (diff)
	    *diff = ovld;
	if (any)
	    *any |= ovld;	/* update anyflag */
	/* check half */
	ovld = (SHORTSIZ16) ((bits & E1432_FLAG_HALF) ? 1 : 0);
	if (half)
	    *half = ovld;
	/* Don't update the any flag for halfrange bit */
    }
    return 0;
}

/*
 *********************************************************************
 Reset request-spectrum flags.  This is called by e1432_reset_measure,
 for example.  It perhaps should be called from e1432_assign_channels,
 but instead we just directly clear the flags when in
 e1432_assign_channels.
 *********************************************************************
 */
SHORTSIZ16
i1432_reset_spectrum(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    int     mod;

    if (ID < 0)
    {
	gn = i1432_get_group_node(hw, ID);
	if (!gn)
	    return i1432_print_error(ERR1432_NO_GROUP);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    gn->modlist[mod]->rs_req = 0;
	    gn->modlist[mod]->rs_ready = 0;
	}
    }
    else
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
	mn->rs_req = 0;
	mn->rs_ready = 0;
    }

    return 0;
}

/* Non-iolib-specific part of irq handler */
static void
generic_irq_handler(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 status;

    /* See if the data really is ready */
    if (!mn->rs_irq_error)
    {
	status = e1432_check_spectrum(mn->hw, mn->chan_id, 0);
	if (status < 0)
	    mn->rs_irq_error = 1;
	if (status > 0)
	    mn->rs_irq_done = 1;
    }

    /* Reenable interrupts if our data is not yet ready */
    if (!mn->rs_irq_done && !mn->rs_irq_error)
    {
	/* For multi-module requests, we need to call
	   e1432_block_available, so that it can take care of it side
	   effects like flushing scans from modules with no channels
	   enabled.  Othersize, we may hang waiting for the spectrum
	   to become ready.  But note that multi-module requests can't
	   be completely reliable even with this. */
	status = e1432_block_available(mn->hw, mn->rs_irq_id);
	if (status < 0)
	    mn->rs_irq_error = 1;

#ifndef	HPVXI_DOWNLOAD
	status = e1432_reenable_interrupt(mn->hw, mn->chan_id);
	if (status)
	    mn->rs_irq_error = 1;
#endif
    }
}

#if	defined(HAVE_SICL) && !defined(EMULATE_SICL)

static void
sicl_irq_handler(INST id, long reason, long sec)
{
    E1432_MODULE_LIST_NODE *mn;
    int     i;
    int     la = sec & E1432_IRQ_STATUS_LADDR_MASK;

    /* Find the module node corresponding to the laddr that interrupted */
    mn = i1432_mod_list;
    for (i = 0; i < i1432_mod_count; i++, mn++)
    {
	if (mn->la == la)
	    break;
    }
    if (i == i1432_mod_count)
    {
	/* Can't find the module that did the interrupt.  No way to
	   return error status, other than a global variable. */
	i1432_rs_irq_error = 1;
	return;
    }

    if (id != mn->rs_irq_sicl_id)
	mn->rs_irq_error = 1;
    if (reason != I_INTR_VXI_SIGNAL)
	mn->rs_irq_error = 1;
    if ((sec & (E1432_IRQ_BLOCK_READY | E1432_IRQ_MEAS_ERROR)) !=
	E1432_IRQ_BLOCK_READY)
	mn->rs_irq_error = 1;

    generic_irq_handler(mn);
}

/* Get the interrupt line to use.  The id parameter is the interface
   session ID. */
static int
sicl_irq_get_line(INST id, int *line)
{
    struct vxiinfo info;
    unsigned long slot0_laddr;
    int     i, error;

    /* Get the slot 0 logical address */
    error = ivxibusstatus(id, I_VXI_BUS_LADDR, &slot0_laddr);
    if (error != 0)
	return -1;

    /* Get resource manager info for slot 0 card */
    error = ivxirminfo(id, slot0_laddr, &info);
    if (error != 0)
	return -1;

    /* Use first available IRQ line that the slot 0 card is handling */
    for (i = 0; i < 8; i++)
	if (info.int_handler[i] != -1)
	{
	    *line = info.int_handler[i];
	    break;
	}
    if (i == 8)
	return -1;

    return 0;
}

static int
sicl_irq_setup(int laddr, INST *irq_id, int *irq_blocked)
{
    char    addr[16];
    int     ierror;

    /* Block SICL interrupts, to avoid getting one before we're ready */
    /* No need to use i1432_introff, this is already SICL-specific. */
    ierror = iintroff();
    if (ierror != 0)
	return -1;
    *irq_blocked = 1;
    /* Open device session */
    (void) sprintf(addr, "vxi,%d", laddr);
    *irq_id = iopen(addr);
    if (*irq_id == 0)
	return -1;
    /* Install SICL interrupt handler */
    ierror = ionintr(*irq_id, sicl_irq_handler);
    if (ierror != 0)
	return -1;
    /* Enable SICL interrupts */
    ierror = isetintr(*irq_id, I_INTR_VXI_SIGNAL, 1);
    if (ierror != 0)
	return -1;

    return 0;
}

static int
sicl_irq_cleanup(INST irq_id, int irq_blocked)
{
    int     ierror;
    int     rv = 0;

    if (irq_id != 0)
    {
	/* Disable SICL interrupts */
	ierror = isetintr(irq_id, I_INTR_VXI_SIGNAL, 0);
	if (ierror != 0)
	    rv = -1;
	/* Uninstall SICL interrupt handler */
	ierror = ionintr(irq_id, NULL);
	if (ierror != 0)
	    rv = -1;
	/* Close device session */
	ierror = iclose(irq_id);
	if (ierror != 0)
	    rv = -1;
    }

    if (irq_blocked)
    {
	/* Unblock SICL interrupts */
	ierror = iintron();
	if (ierror != 0)
	    rv = -1;
    }

    return rv;
}

#endif	/* HAVE_SICL && !EMULATE_SICL */

#ifdef	HAVE_VTL

#include "visa.h"

static int
visa_irq_setup(ViSession vi)
{
    ViStatus status;

    status = viEnableEvent(vi, VI_EVENT_VXI_SIGP, VI_QUEUE, VI_NULL);
    if (status < VI_SUCCESS)
	return -1;

    return 0;
}

static int
visa_irq_cleanup(ViSession vi)
{
    ViStatus status;
    int     rv = 0;

    status = viDisableEvent(vi, VI_EVENT_VXI_SIGP, VI_QUEUE);
    if (status < VI_SUCCESS)
	rv = -1;
    status = viDiscardEvents(vi, VI_EVENT_VXI_SIGP, VI_QUEUE);
    if (status < VI_SUCCESS)
	rv = -1;

    return rv;
}

static int
visa_wait_handler(E1432_MODULE_LIST_NODE *mn)
{
    ViStatus status;
    ViEventType event_type;
    ViEvent event;
    ViUInt16 iack;
    double dtimeout;
    ViUInt32 timeout;

    /* Provide a timeout that is long enough to ensure no errors
       unless the module is somehow hung or broken. */
    if (mn->rs_span == 0)
	dtimeout = 1;		/* Shouldn't happen */
    else
	dtimeout = mn->rs_bsize / (mn->rs_span * 2.56); /* One block */
    if (mn->rs_navg != 0)
	dtimeout *= mn->rs_navg; /* Time for all blocks in average */
    dtimeout = 2 * dtimeout + 2; /* Allow for delays and overhead */
    timeout = (ViUInt32) (1000 * dtimeout + 0.5); /* Convert to ms */

    /* Block until event arrives */
    status = viWaitOnEvent((ViSession) mn->sicl_id, VI_EVENT_VXI_SIGP,
			   timeout, &event_type, &event);
    if (status < VI_SUCCESS)
	return -1;

    /* Get event info */
    status = viGetAttribute(event, VI_ATTR_SIGP_STATUS_ID, &iack);
    if (status < VI_SUCCESS)
	return -1;

    /* Close event context */
    status = viClose(event);
    if (status < VI_SUCCESS)
	return -1;

    /* Consistency checking */
    if (event_type != VI_EVENT_VXI_SIGP)
	return -1;
    if ((iack & E1432_IRQ_STATUS_LADDR_MASK) != mn->la)
	return -1;
    if ((iack & (E1432_IRQ_BLOCK_READY | E1432_IRQ_MEAS_ERROR)) !=
	E1432_IRQ_BLOCK_READY)
	return -1;

    /* Handle the event */
    generic_irq_handler(mn);

    return 0;
}

#endif	/* HAVE_VTL */

static SHORTSIZ16
rs_wait_setup(E1432ID hw, SHORTSIZ16 ID, E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;
    int     line, ierror;

    mn->rs_irq_off = 0;
    mn->rs_irq_save_line = -1;
    mn->rs_irq_save_mask = -1;

    mn->rs_irq_done = 0;
    mn->rs_irq_error = 0;
    i1432_rs_irq_error = 0;

#if	defined(HAVE_SICL) && !defined(EMULATE_SICL)
    mn->rs_irq_sicl_id = 0;

    /* What VME interrupt line should we use? */
    ierror = sicl_irq_get_line(i1432_sicl_id, &line);
    if (ierror)
	return i1432_id_print_error(hw, ID, ERR1432_SICL_ERROR);

    /* Set up SICL interrupts */
    ierror = sicl_irq_setup(mn->la, &mn->rs_irq_sicl_id, &mn->rs_irq_off);
    if (ierror)
	return i1432_id_print_error(hw, ID, ERR1432_SICL_ERROR);
#endif

#if	defined(HAVE_SICL) && defined(EMULATE_SICL)
    /* Don't leave uninitialized */
    line = 1;
#endif

#ifdef	HAVE_VTL
    /* Hardcode to use VME interrupt line one.  VISA does not appear
       to give us a way to query which line to use. */
    line = 1;

    /* Set up VISA interrupts */
    ierror = visa_irq_setup((ViSession) mn->sicl_id);
    if (ierror)
	return i1432_id_print_error(hw, ID, ERR1432_SICL_ERROR);
#endif

#ifdef	HPVXI_DOWNLOAD
    /* Don't leave uninitialized */
    line = 1;
#endif

    /* Save ID for use by handler */
    mn->rs_irq_id = ID;

#ifndef	HPVXI_DOWNLOAD
    /* Reenable E1432 interrupts in case they ended up disabled */
    error = e1432_reenable_interrupt(hw, ID);
    if (error)
	return error;

    /* Save, then set, the interrupt line */
    error = e1432_get_interrupt_priority(hw, ID, &mn->rs_irq_save_line);
    if (error)
	return error;
    error = e1432_set_interrupt_priority(hw, ID, (SHORTSIZ16) line);
    if (error)
	return error;

    /* Save, then set, the interrupt mask */
    error = e1432_get_interrupt_mask(hw, ID, &mn->rs_irq_save_mask);
    if (error)
	return error;
    error = e1432_set_interrupt_mask(hw, ID, E1432_IRQ_BLOCK_READY);
    if (error)
	return error;
#endif

    return 0;
}

static SHORTSIZ16
rs_wait_cleanup(E1432ID hw, SHORTSIZ16 ID, E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error, rv;
    int     ierror;

    rv = 0;

#ifndef	HPVXI_DOWNLOAD
    /* Restore interrupt line setting */
    if (mn->rs_irq_save_line != -1)
    {
	error = e1432_set_interrupt_priority(hw, ID, mn->rs_irq_save_line);
	if (error)
	    rv = error;
    }

    /* Restore interrupt mask setting */
    if (mn->rs_irq_save_mask != -1)
    {
	error = e1432_set_interrupt_mask(hw, ID, mn->rs_irq_save_mask);
	if (error)
	    rv = error;
    }

    /* Reenable E1432 interrupts in case they ended up disabled */
    error = e1432_reenable_interrupt(hw, ID);
    if (error)
	rv = error;
#endif

#if	defined(HAVE_SICL) && !defined(EMULATE_SICL)
    /* Clean up SICL interrupts */
    ierror = sicl_irq_cleanup(mn->rs_irq_sicl_id, mn->rs_irq_off);
    if (ierror != 0)
	rv = i1432_id_print_error(hw, ID, ERR1432_SICL_ERROR);

    mn->rs_irq_sicl_id = 0;
#endif

#ifdef	HAVE_VTL
    /* Clean up VISA interrupts */
    ierror = visa_irq_cleanup((ViSession) mn->sicl_id);
    if (ierror)
	rv = i1432_id_print_error(hw, ID, ERR1432_SICL_ERROR);
#endif

    mn->rs_irq_id = 0;
    mn->rs_irq_off = 0;
    mn->rs_irq_save_line = -1;
    mn->rs_irq_save_mask = -1;

    return rv;
}

static SHORTSIZ16
wait_for_spectrum(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE  *mn;
    SHORTSIZ16 error, rv, status;
    int     ierror;

    rv = 0;

    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;

    /* Setup interrupts */
    error = rs_wait_setup(hw, ID, mn);
    if (error)
    {
	rv = error;
	goto cleanup;
    }

    /* The spectrum might have appeared before interrupts were
       enabled.  Check for that. */
    status = e1432_check_spectrum(hw, ID, 0);
    if (status < 0)
    {
	rv = status;
	goto cleanup;
    }
    if (status > 0)
	goto cleanup;

    /* Use iwaithdlr to unblock interrupts and wait for the handler to
       run.  If the module interrupts for some other reason, we must
       re-do the iwaithdlr. */
    while (!mn->rs_irq_done && !mn->rs_irq_error && !i1432_rs_irq_error)
    {
#if	defined(HAVE_SICL) && !defined(EMULATE_SICL)
	ierror = iwaithdlr(0);
#endif
#if	defined(HAVE_SICL) && defined(EMULATE_SICL)
	ierror = 1;
#endif
#ifdef	HAVE_VTL
	ierror = visa_wait_handler(mn);
#endif
#ifdef	HPVXI_DOWNLOAD
	ierror = 1;
#endif

	if (ierror != 0)
	{
	    rv = i1432_id_print_error(hw, ID, ERR1432_SICL_ERROR);
	    goto cleanup;
	}
    }

    /* If the interrupt handler got an error, report it now */
    if (mn->rs_irq_error || i1432_rs_irq_error)
    {
	rv = i1432_la_print_error(mn->la, ERR1432_SICL_ERROR);
	goto cleanup;
    }

    /* Should not be necessary, but verify that there really is a
       spectrum ready.  This is debug code, remove when everything
       works. */
    status = e1432_check_spectrum(hw, ID, 0);
    if (status < 0)
    {
	rv = status;
	goto cleanup;
    }
    if (status == 0)
    {
	rv = i1432_la_print_error(mn->la, ERR1432_SICL_ERROR);
	goto cleanup;
    }

 cleanup:
    error = rs_wait_cleanup(hw, ID, mn);
    if (error)
	return error;

    return rv;
}

SHORTSIZ16 EXPORT
e1432_check_spectrum(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 block)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    LONGSIZ32 ready;
    int     mod;

    TRACE_PRINTF(0, ("e1432_check_spectrum(0x%p, %d, %d)\n",
		     hw, ID, block));

    if (ID < 0)			/* Group */
    {
	gn = i1432_get_group_node(hw, ID);
	if (!gn)
	    return i1432_print_error(ERR1432_NO_GROUP);

	for (mod = 0; mod < gn->modcount; mod++)
	{
	    error = e1432_check_spectrum(hw,
					 gn->modlist[mod]->chan_id,
					 block);
	    /* If error or no spectrum ready, return now */
	    if (error <= 0)
		return error;
	}
    }
    else			/* Chan */
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;

	/* If no request, then there is no spectrum ready */
	if (mn->rs_req == 0)
	    return 0;

	error = i1432_direct_read32_register(mn,
					     E1432_REQUEST_READY_REG,
					     &ready);
	if (error)
	    return error;

	if (!ready && block)
	{
	    error = wait_for_spectrum(hw, ID);
	    if (error)
		return error;
	    ready = 1;
	}

	if (!ready)
	    return 0;

	mn->rs_ready = 1;
    }

    return 1;
}

/*
 *********************************************************************
 Assumes ID was already verified.
 *********************************************************************
 */
static SHORTSIZ16
check_spectrum_done(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error;
    int     type, count;

    if (ID < 0)			/* Group */
    {
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	while (cn != NULL)
	{
	    error = check_spectrum_done(hw, cn->chanID);
	    if (error)
		return error;
	    cn = cn->next;
	}
    }
    else			/* Chan */
    {
	type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (ID & E1432_CHAN_MASK) - 1;
	if (i1432_chan_list[type][count].active == E1432_CHANNEL_ON)
	{
	    error = i1432_get_module_from_chan(hw, ID, &mn);
	    if (error)
		return error;
	    if (mn->rs_req == 0)
		/* Should not happen */
		return i1432_id_print_error(hw, ID, ERR1432_LIST_ERROR);
	    if (mn->rs_req == 1)
		/* The decrement will take us to zero, so we're done
		   with this module and need to clear rs_ready.  Do
		   before decrementing rs_req so other threads can't
		   start a spectrum before we clear rs_ready. */
		mn->rs_ready = 0;
	    mn->rs_req--;
	}
    }

    return 0;
}

static SHORTSIZ16
verify_spectrum_ready(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod, type, count;

    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    if (ID < 0)			/* Group */
    {
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	    if (gn->modlist[mod]->rs_req == 0 || !gn->modlist[mod]->rs_ready)
		return i1432_id_print_error(hw, gn->modlist[mod]->chan_id,
					    ERR1432_NO_SPECTRUM_READY);
    }
    else			/* Chan */
    {
	type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (ID & E1432_CHAN_MASK) - 1;
	if (i1432_chan_list[type][count].active == E1432_CHANNEL_ON)
	{
	    error = i1432_get_module_from_chan(hw, ID, &mn);
	    if (error)
		return error;
	    if (mn->rs_req == 0 || !mn->rs_ready)
		return i1432_id_print_error(hw, ID, ERR1432_NO_SPECTRUM_READY);
	}
    }

    return 0;
}

SHORTSIZ16 EXPORT
e1432_read_float32_spectrum(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ32 *buf, 
			    LONGSIZ32 size, struct e1432_trailer *trailer, 
			    LONGSIZ32 *actualCnt)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_read_float32_spectrum(0x%p, %d, "
		     "0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, ID, buf, size, trailer, actualCnt));

    error = verify_spectrum_ready(hw, ID);
    if (error)
	return error;

    error = e1432_read_float32_data(hw, ID, E1432_REQUEST_DATA,
				    buf, size, trailer, actualCnt);
    if (error)
	return error;

    error = check_spectrum_done(hw, ID);
    if (error)
	return error;

    return 0;
}

SHORTSIZ16 EXPORT
e1432_read_float64_spectrum(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ64 *buf, 
			    LONGSIZ32 size, struct e1432_trailer *trailer, 
			    LONGSIZ32 *actualCnt)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_read_float64_spectrum(0x%p, %d, "
		     "0x%p, %ld, 0x%p, 0x%p)\n",
		     hw, ID, buf, size, trailer, actualCnt));

    error = verify_spectrum_ready(hw, ID);
    if (error)
	return error;

    error = e1432_read_float64_data(hw, ID, E1432_REQUEST_DATA,
				    buf, size, trailer, actualCnt);
    if (error)
	return error;

    error = check_spectrum_done(hw, ID);
    if (error)
	return error;

    return 0;
}

static SHORTSIZ16
verify_meas_running(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 state, error;

    error = e1432_get_meas_state(hw, ID, &state);
    if (error)
	return error;
    if (state <= E1432_MEAS_STATE_BOOTING ||
	state > E1432_MEAS_STATE_CONVERT_WAIT2)
	return i1432_id_print_error(hw, ID, ERR1432_REQUEST_MEAS_RUNNING);
    return 0;
}

SHORTSIZ16 EXPORT
e1432_request_spectrum(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ32 span,
		       LONGSIZ32 blocksize, LONGSIZ32 avgNum,
		       SHORTSIZ16 avgMode, SHORTSIZ16 window)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_MODULE_LIST_NODE *mn;
    E1432_CHAN_LIST_NODE *cn;
    struct i1432_chan_info *chan_info;
    SHORTSIZ16 mod, error;
    LONGSIZ32 chans;
    int    type, count, magsq, req;

    TRACE_PRINTF(0, ("e1432_request_spectrum(0x%p, %d, "
		     "%g, %ld, %ld, %d, %d)\n",
		     hw, ID, span, blocksize, avgNum, avgMode, window));

    magsq = (avgMode == E1432_AVG_RMS ||
	     avgMode == E1432_AVG_EXPO ||
	     avgMode == E1432_AVG_PEAK);

    if (ID < 0)			/* Group request */
    {
	/* get e1432_group_list_node */
	gn = i1432_get_group_node(hw, ID);
	if (!gn)
	    return i1432_print_error(ERR1432_NO_GROUP);

	cn = gn->chanlist;

	/* for all modules in the group send the request spectrum
	   parameters */
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* get module node */
	    /* !!! This seems bogus */
	    error = i1432_get_module_from_chan(hw, cn->chanID, &mn);
	    if (error)
		return error;

	    /* Abort if no measurement running */
	    error = verify_meas_running(hw, cn->chanID);
	    if (error)
		return error;

	    /* Abort if spectrum already requested, or zoom is on */
	    if (mn->rs_req > 0)
		return i1432_id_print_error(hw, cn->chanID,
					    ERR1432_PRIOR_SPECTRUM_REQUESTED);
	    if (mn->zoom)
		return i1432_id_print_error(hw, cn->chanID,
					    ERR1432_REQUEST_ZOOM);

	    /* Set rs_req non-zero, to prevent other threads from
	       requesting a spectrum.  It'll get the correct value
	       later, but must be non-zero now.  There's a window
	       between the above check and this set, which should be
	       protected with a mutex. */
	    mn->rs_req = 1;

	    /* If not doing order tracking, tell the module to switch
	       to the requested span */
	    if (!mn->order_tracking)
	    {
		error = e1432_set_span(hw, mn->chan_id, span);
		if (error)
		    return error;
	    }

	    chans = 0;
	    req = 0;

	    /* !!! This loop seems bogus */
	    while(cn)
	    {
		/* read data for active channels */
		type = (cn->chanID & E1432_CHAN_TYPE_MASK) >>
		    E1432_CHAN_TYPE_SHIFT;
		count = (cn->chanID & E1432_CHAN_MASK) -1;

		/* check if we've gotten all the channels for the current module */
		if (i1432_chan_list[type][count].mn != mn)
		    break;
		else
		{
		    /* create bitfield for all chans in module which are active */
		    if (i1432_chan_list[type][count].active ==
			E1432_CHANNEL_ON)
		    {
			chans |= 1 << i1432_chan_list[type][count].index;
			req++;
		    }

		    /* next channel */
		    cn = cn->next;
		}
	    }

	    if (req == 0)
		return 0;

	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_CHANS_REG,
						  chans);
	    if (error)
		return error;
	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_SPAN_REG,
						  *(LONGSIZ32 *)(&span));
	    if (error)
		return error;
	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_BLOCKSIZE_REG,
						  blocksize);
	    if (error)
		return error;
	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_AVG_NUM_REG,
						  avgNum);
	    if (error)
		return error;
	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_AVG_MODE_REG,
						  avgMode);
	    if (error)
		return error;
	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_WINDOW_REG,
						  window);
	    if (error)
		return error;
	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_READY_REG,
						  0);
	    if (error)
		return error;
	    error = i1432_direct_write32_register(mn,
						  E1432_REQUEST_START_REG,
						  1);
	    if (error)
		return error;

	    mn->rs_bsize = blocksize;
	    mn->rs_navg = avgNum;
	    mn->rs_span = span;
	    mn->rs_magsq = magsq;
	    mn->rs_ready = 0;
	    mn->rs_req = req;
	}
    }
    else			/* Chan request */
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return i1432_print_error(error);

	/* Abort if no measurement running */
	error = verify_meas_running(hw, ID);
	if (error)
	    return error;

	/* Abort if spectrum already requested, or zoom is on */
	if (mn->rs_req > 0)
	    return i1432_id_print_error(hw, ID,
					ERR1432_PRIOR_SPECTRUM_REQUESTED);
	if (mn->zoom)
	    return i1432_id_print_error(hw, ID,
					ERR1432_REQUEST_ZOOM);

	/* Set rs_req non-zero, to prevent other threads from
	   requesting a spectrum.  There's a window between the above
	   check and this set, which should be protected with a
	   mutex. */
	mn->rs_req = 1;

	/* If not doing order tracking, tell the module to switch
	   to the requested span */
	if (!mn->order_tracking)
	{
	    error = e1432_set_span(hw, ID, span);
	    if (error)
		return error;
	}

	count = (ID & E1432_CHAN_MASK) - 1;
	chan_info = i1432_chan_list[E1432_CHAN_TYPE_INPUT];

	/* Set bitfield for chan in module */
	chans = 0;
	if (chan_info[count].mn == mn &&
	    chan_info[count].active == E1432_CHANNEL_ON)
            chans |= 1 << chan_info[count].index;
	else
	    return i1432_print_error(ERR1432_NO_CHANNEL);

	/* Send the request spectrum parameters */
	error = i1432_direct_write32_register(mn, E1432_REQUEST_CHANS_REG,
					      chans);
	if (error)
	    return error;
	error = i1432_direct_write32_register(mn, E1432_REQUEST_SPAN_REG,
					      *(LONGSIZ32 *) (&span));
	if (error)
	    return error;
	error = i1432_direct_write32_register(mn, E1432_REQUEST_BLOCKSIZE_REG,
					      blocksize);
	if (error)
	    return error;
	error = i1432_direct_write32_register(mn, E1432_REQUEST_AVG_NUM_REG,
					      avgNum);
	if (error)
	    return error;
	error = i1432_direct_write32_register(mn, E1432_REQUEST_AVG_MODE_REG,
					      avgMode);
	if (error)
	    return error;
	error = i1432_direct_write32_register(mn, E1432_REQUEST_WINDOW_REG,
					      window);
	if (error)
	    return error;
	error = i1432_direct_write32_register(mn, E1432_REQUEST_READY_REG,
					      0);
	if (error)
	    return error;
	error = i1432_direct_write32_register(mn, E1432_REQUEST_START_REG,
					      1);
	if (error)
	    return error;

	mn->rs_bsize = blocksize;
	mn->rs_navg = avgNum;
	mn->rs_span = span;
	mn->rs_magsq = magsq;
	mn->rs_ready = 0;
    }
    return 0;
}

/*
 *********************************************************************
 Transfer user window directly to the Y memory window location and notify
 the downloadable that there is a user window with a user scale factor.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_set_user_window(E1432ID hw, SHORTSIZ16 ID, 
		      FLOATSIZ64 *window, FLOATSIZ64 scale)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error2;
    LONGSIZ32 start, length, page;
    float   *buf;
    int     i;
    float   fscale = (float) scale;

    if (ID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    buf = malloc(E1432_WINDOW_SIZE_MAX * sizeof (float));
    if (buf == NULL)
    {
	error = i1432_print_error(ERR1432_MALLOC);
	goto cleanup;
    }

    for (i = 0; i < E1432_WINDOW_SIZE_MAX; i++)
	buf[i] = (float) window[i];

    error = e1432_set_window(hw, ID, E1432_WINDOW_USER1);
    if (error)
	goto cleanup;

    error = i1432_write_cmd2(hw, ID, E1432_CCMD_SET_WINDOW_SCALE, 
			     0, *(LONGSIZ32 *)&fscale);
    if (error)
	goto cleanup;

    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	goto cleanup;

    start = E1432_WINDOW_START;
    length = E1432_WINDOW_SIZE_MAX;

#if SWAP
    if (!mn->d32)
    {
	/* The window is 32-bit data.  If the data will get
	   transferred (and byte swapped) in 16-bit chunks, we need to
	   pre-swap the data now to fix it. */
	error = i1432_two_four_swap((char *) buf, length * 4);
	if (error)
	    goto cleanup;
    }
#endif

    page = E1432_B_SRAM_PAGE;
    if (mn->a24_256k)
	page <<= 2;

    error = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 
					(SHORTSIZ16) page);
    if (error)
	goto cleanup;

    error = i1432_xfer_block(mn, (unsigned long *) buf,
			     (unsigned long *) start,
			     length, 0, 0);

 cleanup:
    error2 = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 
					 E1432_BASE_PAGE);
    if (error2)
	error = error2;

    free(buf);

    return error;
}


/*
 *********************************************************************
 Read the order current value table for the channel(s) specified by
 <ID> and put the result into the buffer pointed to by <buffer>. 
 <points> is the number of points to get for each channel in <ID>.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_get_cvt_data(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ32 *buffer, int points)
{
    SHORTSIZ16 error;
    int chan, chan_mask, chan_num, chan_bit;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_MODULE_LIST_NODE *mn;
    FLOATSIZ32 *in, *out;
    char err[80];

    /* check if real ID */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;
    
    /* request a lock on the CVT for reading */
    error = i1432_direct_write32_register(mn, E1432_CVT_LOCK_REQ_REG, 1);
    if (error)
	goto cleanup;

    /* wait for downloadable to lock the CVT */
    (void) sprintf(err, "module at %d never locked CVT", mn->la);
    error = i1432_wait32_bits_match(mn, E1432_CVT_LOCKED_REG, 1, 1, 
							CVT_TIMEOUT, err);
    if(error)
	goto cleanup;

    /* set bits in channel mask equal to channel numbers in ID */
    if (ID < 0)		/* group of channels */
    {
        gn = i1432_get_group_node(hw, ID);
        if (!gn)
	{
	    error = i1432_print_error(ERR1432_NO_GROUP);
	    goto cleanup;
	}

	if (gn->modcount > 1)
	{
	    /* The code below doesn't handle groups with more than one
	       module in them, so error rather than produce bogus
	       results. */
	    error = i1432_print_error(ERR1432_REQUIRE_ONE_MOD);
	    goto cleanup;
	}

        cn = gn->chanlist;  /* channel list for group */

        /* check each channel */
	chan_mask = 0;
	chan_num = -1;
        while (cn)
	{
	    chan = i1432_get_chan_index(hw, cn->chanID);
	    /* Was: chan = (cn->chanID & E1432_CHAN_MASK) - 1; */

	    /* set bitfield for chan in module */
	    chan_mask |=  1 << chan;
	    cn = cn->next;

	    /* Save max channel number for use in list below.  The
	       channels in a group are guaranteed to be in increasing
	       order, so the last one will be the maximum one. */
	    chan_num = chan;
	}
    }
    else		/* single channel */
    {
	chan = i1432_get_chan_index(hw, ID);
	/* Was: chan = (ID & E1432_CHAN_MASK) - 1; */

	/* set bitfield for chan in module */
	chan_mask =  1 << chan;
	chan_num = chan;
    }

    /* use channel mask to read appropriate channel's CVT into buffer */
    if(points > 2 * E1432_ORDER_CVT_SIZE)
	points = 2 * E1432_ORDER_CVT_SIZE;

    chan_bit = 1;
    chan = 0;
    in = (float *)E1432_ORDER_CVT;
    out = buffer;

    for (chan = 0; chan <= chan_num; chan++)
    {
	/* read CVT channel by channel */
	if(chan_bit & chan_mask)  /* if a channel in <ID> */
	{
	    error = i1432_xfer_block(mn, (unsigned long *)in,
				(unsigned long *)out, points, 0, 1);
	    if(error)
	        goto cleanup;
#if SWAP
	    if (!mn->d32)
	    {
		/* The data is 32-bit data.  If the data was
		   transferred (and byte swapped) in 16-bit chunks, we
		   need to swap the data now to fix it. */
		error = i1432_two_four_swap((char *) out, points * 4);
		if (error)
		    goto cleanup;
	    }
#endif

	    out += points;
	}
	chan_bit <<= 1;
	in += 2 * E1432_ORDER_CVT_SIZE;
    }

cleanup:
    /* always leave with these handshake registers clear */
    (void)i1432_direct_write32_register(mn, E1432_CVT_LOCK_REQ_REG, 0);
    (void)i1432_direct_write32_register(mn, E1432_CVT_LOCKED_REG, 0);

    return error;
}
